/*
 * SoundManager.cpp
 *
 *  Created on: 15.07.2011
 *      Author: stefan.detter
 */

#include "Sound.h"
#include "SoundManager.h"

#if USE_SOUND

#include <QSound>
#include <QDateTime>
#include <QMessageBox>
#include <QApplication>

#include "../Resource.h"
#include "SoundPlayer.h"

SoundManager* SoundManager::d = 0;

SoundManager::SoundManager(QObject* parent)
	: QObject(parent)
{
	connect(Resource::d, 	SIGNAL( addFileRequest(const QFileInfo&, bool&, bool&, QString&) ),
			this, 			  SLOT( addFileRequest(const QFileInfo&, bool&, bool&, QString&) ), Qt::DirectConnection);

	m_soundPlayer = new SoundPlayer(this);

	reloadFiles();
}

SoundManager::~SoundManager()
{
	delete m_soundPlayer;
}

void SoundManager::playSound(int index)
{
	m_soundPlayer->play(index);
}

QStringList SoundManager::playlist()
{
	return QStringList() << m_files.keys();
}

uint SoundManager::parallelCount()
{
	return m_parallelCount;
}

void SoundManager::setParallelCount(uint count)
{
	m_parallelCount = count;
	m_soundPlayer->setParallelCount(m_parallelCount);
}

void SoundManager::addFileRequest ( const QFileInfo& file, bool& yesToAll, bool& result, QString& errMsg )
{
	if(result == true)
		return;

	if(!checkFile(file, errMsg))
	{
		result = false;
		return;
	}

	if(m_files.keys().contains(file.fileName()))
	{
		if(!yesToAll)
		{
			QMessageBox::StandardButton b;
			b = QMessageBox::question(QApplication::activeWindow(),
									  "Add Sound File",
									  QString(
											  "The sound repository already contains this sound file.\n\n"
											  "Old file from: %1\n"
											  "New file from: %2\n\n"
											  "Do you want to replace the firmware file?")
									  .arg(m_files.value(file.fileName()).lastModified().toString())
									  .arg(file.lastModified().toString()),
									  QMessageBox::Yes | QMessageBox::No | QMessageBox::YesToAll,
									  QMessageBox::Yes
									  );

			if(b == QMessageBox::No)
			{
				errMsg = "File should not be replaced.";
				result = false;
				return;
			} 
			else if(b == QMessageBox::YesToAll)
			{
				yesToAll = true;
			}
		}

		QFile::remove(m_files.value(file.fileName()).absoluteFilePath());
		m_files.remove(file.fileName());
	}

	QFileInfo newFile;
	if(!Resource::d->copyFileToResource(file, Resource::SoundResource, "", newFile))
	{
		errMsg = "The sound file could not be copied.";
		result = false;
		return;
	}

	reloadFiles();

	errMsg = "The sound file was successfully added to the repository.";
	result = true;
	return;
}

void SoundManager::initInstance(QObject* parent)
{
	d = new SoundManager(parent);
}



bool SoundManager::reloadFiles()
{
	QFileInfoList fileList = Resource::d->soundFiles();
	QStringList filePaths;
	m_files.clear();

	foreach(QFileInfo info, fileList)
	{
		m_files.insert(info.fileName(), info);
		filePaths.append(info.absoluteFilePath());
	}

	return m_soundPlayer->setFiles(filePaths);
}

bool SoundManager::checkFile ( const QFileInfo& file, QString& errMsg )
{
	if(!file.exists())
	{
		errMsg = "The file does not exist.";
		return false;
	}

	if(!file.isFile())
	{
		errMsg = "The data type is not a file.";
		return false;
	}

	if(!file.isReadable())
	{
		errMsg = "The file is not readable.";
		return false;
	}

	if(file.suffix() != "wav")
	{
		errMsg = "Unkown file type.";
		return false;
	}

	return true;
}

#else

SoundManager* SoundManager::d = 0;

SoundManager::SoundManager(QObject* parent)
	: QObject(parent)
{

}

SoundManager::~SoundManager()
{
}

QStringList SoundManager::playlist()
{
	return QStringList();
}

uint SoundManager::parallelCount()
{
	return m_parallelCount;
}

void SoundManager::setParallelCount(uint count)
{
	m_parallelCount = count;
}

void SoundManager::addFileRequest ( const QFileInfo& file, bool& yesToAll, bool& result, QString& errMsg )
{
	return;
}

void SoundManager::playSound(int index)
{
	return;
}

void SoundManager::initInstance(QObject* parent)
{
	d = new SoundManager(parent);
}

#endif
